#!/usr/bin/perl -w

# MAYA SUBMIT GUI
#
#     Rush 102.42a Render Queue Software.		##RUSH_VERSION##
#     (C) Copyright Greg Ercolano, 2002. All rights reserved.
#
#     This is a sample Maya GUI submit script you can customize
#     for your company's own purposes.
#
use strict;
use FindBin;

$G::title          = "Rush Maya Submit";
$G::progname       = "submit-maya";
$G::logtrim_always = 0;		# 1=always use logtrim (improves render times)

# LOAD COMMON VARIABLES + FUNCTIONS
require "$FindBin::Bin/.common.pl";

# CUSTOMIZE IF NECESSARY
@G::projsep = ( "/renderscenes/", "/scenes/", "/scn/" );    # project path components

#########################################################
### MAYA SPECIFIC VARIABLES -- CUSTOMIZE AS NECESSARY ###
### See Maya docs for more info.                      ###
#########################################################
if ( $G::iswindows )
{
    ### WINDOWS -- should already be in System Environment; leave commented out unless needed.
    # $ENV{MAYA_LOCATION} = "c:/Program Files/AliasWavefront/Maya5.0";	# MAYA5.0
    # $ENV{MAYA_LOCATION} = "c:/Program Files/Alias/Maya6.0";		# MAYA6.0
    # $ENV{MAYA_LOCATION} = "c:/Program Files/Alias/Maya6.5";		# MAYA6.5
    # $ENV{PATH}          = "$ENV{MAYA_LOCATION}/bin;$ENV{PATH}";

    # Uncomment for Mentalray Standalone
    # $ENV{MI_ROOT}    = "c:/program files/alias/mentalray3.4";
    # $ENV{MI_BIN_DIR} = "$ENV{MI_ROOT}/bin";
    # $ENV{PATH}       = "$ENV{MI_BIN_DIR};$ENV{PATH}";
}
elsif ( $G::ismac )
{
    ### MAC
    $ENV{PATH} = "/Applications/Alias/maya6.5/Maya.app/Contents/bin:".  # try maya 6.5 first
                 "/Applications/Alias/maya6.0/Maya.app/Contents/bin:".
                 "/usr/local/bin:/usr/sbin:$ENV{PATH}";

    # These are optional.. Maya picks good defaults
    # $ENV{MAYA_LOCATION} = "/Applications/AliasWavefront/maya5.0/Maya.app/Contents"; # MAYA5.0
    # $ENV{MAYA_LOCATION} = "/Applications/Alias/maya6.0/Maya.app/Contents";          # MAYA6.0
    # $ENV{MAYA_LOCATION} = "/Applications/Alias/maya6.5/Maya.app/Contents";          # MAYA6.5

    # Uncomment for Mentalray Standalone
    # $ENV{MI_ROOT}    = "/usr/local/mi";
    # $ENV{MI_BIN_DIR} = "$ENV{MI_ROOT}/bin";
    # $ENV{PATH}       = "$ENV{MI_BIN_DIR}:$ENV{PATH}";
}
elsif ( $G::islinux )
{
    ### LINUX
    $ENV{PATH} = "/usr/aw/maya6.5/bin:".		# try maya 6.5 first
                 "/usr/aw/maya6.0/bin:".
		 "/usr/aw/maya5.0/bin:$ENV{PATH}";

    # These are optional.. Maya picks good defaults
    # $ENV{MAYA_LOCATION} = "/usr/aw/maya5.0";		# MAYA5.0
    # $ENV{MAYA_LOCATION} = "/usr/aw/maya6.0";		# MAYA6.0
    # $ENV{MAYA_LOCATION} = "/usr/aw/maya6.5";		# MAYA6.5

    # Uncomment for Mentalray Standalone
    # $ENV{MI_ROOT}    = "/usr/local/mi";
    # $ENV{MI_BIN_DIR} = "$ENV{MI_ROOT}/bin";
    # $ENV{PATH}       = "$ENV{MI_BIN_DIR}:$ENV{PATH}";
}
elsif ( $G::isirix )
{
    ### IRIX
    $ENV{PATH} = "/usr/aw/maya6.5/bin:".		# try maya 6.5 first
                 "/usr/aw/maya6.0/bin:".
		 "/usr/aw/maya5.0/bin:$ENV{PATH}";

    # These are optional.. Maya picks good defaults
    # $ENV{MAYA_LOCATION} = "/usr/aw/maya5.0";		# MAYA5
    # $ENV{MAYA_LOCATION} = "/usr/aw/maya6.0";		# MAYA6
    # $ENV{MAYA_LOCATION} = "/usr/aw/maya6.5";		# MAYA6.5

    # Uncomment for Mentalray Standalone
    # $ENV{MI_ROOT}    = "/usr/local/mi";
    # $ENV{MI_BIN_DIR} = "$ENV{MI_ROOT}/bin";
    # $ENV{PATH}       = "$ENV{MI_BIN_DIR}:$ENV{PATH}";
}

### Other MAYA variables you may want to set.
#   $ENV{MAYA_PLUG_IN_PATH}        = "$ENV{MAYA_LOCATION}/bin/plug-ins";
#   $ENV{MAYA_SHADER_LIBRARY_PATH} = "$ENV{MAYA_LOCATION}/shaders";
#   $ENV{MAYA_MODULE_PATH}         = "$ENV{MAYA_LOCATION}/modules";

### Other MENTALRAY STANDALONE variables you may want to set
#   $ENV{MI_LIBRARY_PATH}          = "$ENV{MAYA_LOCATION}/mentalray/lib";
#   $ENV{MI_RAY_INCPATH}           = "$ENV{MAYA_LOCATION}/mentalray/include";
#   $ENV{SPM_HOST}                 = "some-hostname";    # mray standalone license server hostname

# RETURN PATHNAME TO RENDERED IMAGE
#    $1: logile to try to grep rendered image name from
#    RETURNS path name to image, or "-" if unable to determine.
#
sub GetRenderedImage($)
{
    my ($logfile) = @_;
    my $image = "-";
    my $imagedir = "-";
    unless ( open(FD, "<$logfile" ) )
	{ ErrorWindow("$logfile: $!"); exit(1); }
    while ( <FD> )
    {
	chomp();

	# MAYA:
	# Finished Rendering //server/vol/images/test.0010.gif
	if ( /^Finished Rendering (\S+)/ )	# rendered image pathname
	    { $image = $1; $image =~ s/\.$//; }

	# MENTAL RAY:
	# PHEN 0.0  progr: writing image file //server/foo/x.0005.rgb (frame 5)
	if ( /: writing image file (\S+)/ )	# rendered image pathname
	    { $image = $1; $image =~ s/\.$//; }

	# MENTAL RAY
	#     IMAGEDIR: /server/volume/project/images 
	if ( /^[\s+]*IMAGEDIR: (\S+)/ )
	    { $imagedir = $1 }
    }
    close(FD);

    # XXX: Sometimes mray doesn't get an absolute path to the image, eg:
    # PHEN 0.0  progr: writing image file xray_test.0003.tif (frame 3)
    #                                     ^^^^^^^^^^^^^^^^^^
    # ..so try inserting the $imagedir first.
    #
    if ( ! -e $image && $imagedir ne "-" )
        { $image = "$imagedir/$image"; }

    return($image);
}

# SAVE OUT DEFAULT FORM
sub WriteDefaults($)
{
    my $filename = $_[0];
    unless ( open(DEFAULTS, ">$filename") )
        { return("$filename: $!"); }

    print DEFAULTS <<"EOF";
         JobTitle: 
        ScenePath: //server/jobs/SHOW/SHOT${G::projsep[0]}test.mb
   ImageDirectory: 
     ImageCommand: 
     LogDirectory: 
           Frames: 1-10
      BatchFrames: 1
        BatchClip: yes
          MaxTime: 00:00:00
     MaxTimeState: Que
       MaxLogSize: 0
             Cpus: +any=5
              Ram: 1
         AutoDump: off
         DoneMail:
         DumpMail:
          WaitFor: 
     WaitForState: Dump
        MayaFlags:
     NeverUseCpus: 
    SubmitOptions: 
 	 LogFlags: Overwrite
      SubmitHosts: 
  JobStartCommand:
   JobDoneCommand:
   JobDumpCommand:
       StartIrush: yes
 PrintEnvironment: off
          Retries: 3
    RetryBehavior: Fail
         Renderer: maya
      MayaVerbose: 1
  MayatomrVerbose: 5
    MayatomrFlags:
   VectorizeFlags:
      MrayVerbose: 5
        MrayFlags:
  SaveMIDirectory: 
            Debug: off
EOF
    close(DEFAULTS);
    return("");
}

# RUN MAYA
#    Check for exit codes and license errors
#    Returns: 0=OK, 1=failed, 2=retry
#       $1 - command to invoke
#       $2 - name of renderer
#
sub RunMaya($$)
{
    my ($command,$renderer) = @_;
    print "\nExecuting: $command\n";
    my $errmsg;
    my $exitcode = RunCommand($command, \$errmsg);
    print STDERR "\n";

    if ( $exitcode == 128 && $G::iswindows )		# MAYA LICENSE ERROR
    {
	print STDERR
	    "--- DETECTED WINDOWS MAYA LICENSE BUG ('EXIT 128')\n".
	    "--- See http://seriss.com/rush-current/rush/maya-issues.html\n".
	    "--- ADDING THIS MACHINE AS NEVERHOST AND REQUEING\n";
	system("rush -fu -an $ENV{RUSH_HOSTNAME}");	# NEVERHOST
	system("rush -fu -jobnotes \"EXIT 128: $ENV{RUSH_PADFRAME} " .
	       "on $ENV{RUSH_HOSTNAME}: Added as neverhost/requeue\"");
	exit(2);					# REQUEUE
    }
    elsif ( $exitcode > 0 && $exitcode < 128 )		# NON-ZERO EXIT
	{ print STDERR "--- $renderer FAILED: EXITCODE=$exitcode\n"; return(1); }
    elsif ( $exitcode != 0 )				# CMDFAIL/SEGFAULT/SIG
	{ print STDERR "--- $errmsg\n"; return(1); }

    # NO EXIT CODE: LICENSE CHECK
    if ( ( $errmsg = LogCheck("^Executing: ", 
                                 ( "could not get a license",
				   "no license available",
				   "to any SPM license server") ) ) ne "" )
    {
        print STDERR "--- $renderer LICENSE ERROR: Encountered: $errmsg\n";
	return(2);	# retry, not requeue
    }

    print STDERR "$renderer SUCCEEDS: EXITCODE=$exitcode\n";
    return(0);
}

# CREATE A MEL SCRIPT TO VECTOR RENDER WITH MAYA
#    $1 - mel script to create
#    $2 - sfrm
#    $3 - efrm
#
#    Returns -1 on error; error message will have been 
#    already printed on stderr.
#
sub CreateVectorMel($%)
{
    my ( $melfile, %opt) = @_;
    unless ( open(MEL, ">$melfile") )
        { print STDERR "ERROR: $melfile: $!\n"; return(-1); }

    # This mel code could be a *lot* simpler if we knew how
    # to override the imagedir it generates for vectorize's "-of".
    #
    # The simpler version would be:
    #     source "vectorRenderUtil.mel";
    #     loadPlugin "VectorRender";
    #     setAttr "defaultRenderGlobals.startFrame"  $opt{sfrm};
    #     setAttr "defaultRenderGlobals.endFrame"    $opt{efrm};
    #     setAttr "defaultRenderGlobals.byFrameStep" 1;
    #     vectorRenderInBatchMode();
    #
    print MEL <<"EOF";

////////////////////////////////////////////////////// VECTOR MEL SCRIPT: START

source "vectorRenderUtil.mel";

// Return default camera
global proc string RUSH_DefaultCamera()
{
    string \$camshape = vrFirstRenderableCamera();
    string \$camv[] = `listTransforms \$camshape`;
    string \$cam;
    if ( size( \$camv ) > 0 )
        { \$cam = \$camv[0]; }
    // DEBUGGING
    if ( "$opt{Debug}" == "on" )
    {
        print("*** CAMERASHAPE=" + \$camshape + "\\n");
        print("***   CAMV SIZE=" + size(\$camv) + "\\n");
        int \$i;
	for ( \$i=0; \$i<size(\$camv); \$i++ )
	    { print("***     CAMV["+\$i+"]="+\$camv[\$i]+"\\n"); }
    }
    return( \$cam );
}

// See vectorRenderExecuteBatch.mel
if ( !`pluginInfo -q -l "VectorRender"` )
    { loadPlugin "VectorRender"; }

// Only render start/end/increment
setAttr "defaultRenderGlobals.startFrame"  $opt{sfrm};
setAttr "defaultRenderGlobals.endFrame"    $opt{efrm};
setAttr "defaultRenderGlobals.byFrameStep" 1;

// Make project the cwd
string \$project = (`workspace -q -fn`) + "/";
if ( chdir( \$project ) != 0 )
    { error("chdir(" + \$project + ": failed\\n"); }

// Verify scene file exists
string \$scenefile = `file -q -sn`;
if ( size(\$scenefile) == 0 )
    { error(\$scenefile + ": scene file does not exist or not specified\\n"); }

// Build vectorize command
string \$cam  = RUSH_DefaultCamera();
string \$xres = `getAttr defaultResolution.width`;
string \$yres = `getAttr defaultResolution.height`;
string \$cmd  = vrVectorBuildCommand(false, \$xres, \$yres, \$cam);

// User specified imagedir?
//     Replace 'vectorize' command's "-of" pathname with user's own.
//
if ( "$opt{ImageDirectory}" != "-" )
{
    string \$imagedir = "$opt{ImageDirectory}";
    if ( match("[/\\\\]\$", \$imagedir) == "" )
        { \$imagedir = \$imagedir + "/" + vrGetImageNamePrefix(); }
    \$cmd = substitute("-of \\".*\\"", \$cmd, ("-of \\""+\$imagedir+"\\"")); 
}

print("Executing: " + \$cmd + "\\n");
eval(\$cmd);

////////////////////////////////////////////////////// VECTOR MEL SCRIPT: END

EOF
    close(MEL);
    return(0);
}

# RENDER USING VECTOR RENDERER
#     Returns exit codes: 0 if ok, 1 on error.
#     (May exit on its own with 2 to force requeue.)
#
sub RenderVector(%)
{
    my (%opt) = @_;

    my $melscript = "$ENV{RUSH_TMPDIR}/foo.mel";  # path to tmp mel script

    # PRINT WHAT WE'RE RENDERING
    print STDERR "\n*** VECTOR RENDER ***\n".
		 "   RENDERER: $opt{Renderer}\n".
		 "  MAYAFLAGS: $opt{MayaFlags}\n".
                 "  SCENEPATH: $opt{ScenePath}\n".
		 "    PROJECT: $opt{Project}\n".
		 "   IMAGEDIR: $opt{ImageDirectory}\n".
		 "BATCHFRAMES: $opt{BatchFrames} ($opt{sfrm}-$opt{efrm})\n".
		 "    RETRIES: $opt{Retries} ".
			 "($opt{RetryBehavior} after $opt{Retries} retries)\n".
		 " MAXLOGSIZE: $opt{MaxLogSize}\n".
		 "       PATH: $ENV{PATH}\n";

    # Create a temporary mel script used to generate the mi files
    CreateVectorMel($melscript, %opt);

    # DEBUGGING: PRINT OUT THE MEL SCRIPT
    print CatFile($melscript);

    # MAYA MI-GEN COMMAND
    my $command = (($G::iswindows) ? "mayabatch " : "maya -batch ") .
		  "-proj $opt{Project} ".
		  "-script $melscript ".
		  "$opt{MayaFlags}".
		  "-file $opt{ScenePath}";

    # HANDLE TRIMMING LOG SIZE
    if ( $opt{MaxLogSize} > 0 || $G::logtrim_always )
	{ $command = "logtrim -s $opt{MaxLogSize} -c $command"; }

    return(RunMaya($command, "MAYA-VECTOR"));
}

# CREATE A MEL SCRIPT TO RENDER MRAY
#    $1 - mel pathname, file to be created
#    $2 - %opt hash
#
#    Returns -1 on error; error message will have been 
#    already printed on stderr.
#
sub CreateMrayMel($%)
{
    my ( $melfile, %opt) = @_;
    unless ( open(MEL, ">$melfile") )
        { print STDERR "ERROR: $melfile: $!\n"; return(-1); }
    print MEL <<"EOF";

//////////////////////////////////////////////////////// MRAY MEL SCRIPT: START

// Image dir specified? Use it.
if ( "$opt{ImageDirectory}" != "-" )
{
    if ( chdir( "$opt{ImageDirectory}" ) != 0 )
	{ error("chdir($opt{ImageDirectory}): failed\\n"); }

    // Force Mayatomr to use our ImageDirectory. -erco 02/09/04
    setAttr "mentalrayGlobals.outp" -type "string" "$opt{ImageDirectory}";
}

if ( !`pluginInfo -q -l "Mayatomr"` )
    { error("Mayatomr not found. Make sure 'MAYA_PLUG_IN_PATH' set correctly."); }

// Scene file existence check
string \$scene = `file -q -sn`;
if ( size( \$scene ) == 0 )
    { error ("'"+\$scene+"': scene file not found or not specified\\n\\n"); }

// Initialize Mayatomr globals
miCreateDefaultNodes();

// Init mray globals
//
// Maya 6.x and older                                  // Maya 7.x and up
// ------------------                                  // ---------------
setAttr "mentalrayGlobals.startFrame"     $opt{sfrm};  setAttr "defaultRenderGlobals.startFrame"     $opt{sfrm};
setAttr "mentalrayGlobals.endFrame"       $opt{efrm};  setAttr "defaultRenderGlobals.endFrame"       $opt{efrm};
setAttr "mentalrayGlobals.byFrame"        1;           setAttr "defaultRenderGlobals.byFrame"        1;
setAttr "mentalrayGlobals.startExtension" $opt{sfrm};  setAttr "defaultRenderGlobals.startExtension" $opt{sfrm};
setAttr "mentalrayGlobals.byExtension"    1;           setAttr "defaultRenderGlobals.byExtension"    1;

Mayatomr -render -v $opt{MayatomrVerbose} $opt{MayatomrFlags};

//////////////////////////////////////////////////////// MRAY MEL SCRIPT: END

EOF
    close(MEL);
    return(0);
}

# RENDER USING MENTAL RAY
#     Returns exit codes: 0 if ok, 1 on error, 2 to requeue
#     (May exit on its own with 2 to force requeue.)
#
sub RenderMray(%)
{
    my (%opt) = @_;
    my $melscript = "$ENV{RUSH_TMPDIR}/foo.mel";	# path to tmp mel script

    # PRINT WHAT WE'RE RENDERING
    print STDERR "\n*** MAYA MENTAL RAY ***\n".
		 "   RENDERER: $opt{Renderer}\n".
		 "  MAYAFLAGS: $opt{MayaFlags}\n".
                 "  SCENEPATH: $opt{ScenePath}\n".
		 "    PROJECT: $opt{Project}\n".
		 "   IMAGEDIR: $opt{ImageDirectory}\n".
		 "BATCHFRAMES: $opt{BatchFrames} ($opt{sfrm}-$opt{efrm})\n".
		 "    RETRIES: $opt{Retries} ".
			 "($opt{RetryBehavior} after $opt{Retries} retries)\n".
		 " MAXLOGSIZE: $opt{MaxLogSize}\n".
		 "       PATH: $ENV{PATH}\n";

    # Create a temporary mel script used to generate the mi files
    CreateMrayMel($melscript, %opt);

    # DEBUGGING: PRINT OUT THE MEL SCRIPT
    print CatFile($melscript);

    # MAYA RENDER COMMAND
    my $command = (($G::iswindows) ? "mayabatch " : "maya -batch ") .
		  "-proj $opt{Project} ".
		  "-script $melscript ".
		  "$opt{MayaFlags}".
		  "-file $opt{ScenePath}";

    # HANDLE TRIMMING LOG SIZE
    if ( $opt{MaxLogSize} > 0 || $G::logtrim_always )
	{ $command = "logtrim -s $opt{MaxLogSize} -c $command"; }

    unless ( chdir($opt{Project}) )
        { print "chdir($opt{Project}): $!\n"; return(1); }

    return(RunMaya($command, "MAYA-MRAY"));
}

# CREATE A MEL SCRIPT TO RENDER MRAY IN STANDALONE MODE
#    $1 - mel pathname, file to be created
#    $2 - %opt hash
#
#    Returns -1 on error; error message will have been 
#    already printed on stderr.
#
sub CreateMrayStandaloneMel($%)
{
    my ( $melfile, %opt) = @_;
    unless ( open(MEL, ">$melfile") )
        { print STDERR "ERROR: $melfile: $!\\n"; return(-1); }
    print MEL <<"EOF";

//////////////////////////////////////////////////////// MRAY STANDALONE MEL SCRIPT: START

if ( !`pluginInfo -q -l "Mayatomr"` )
    { error("Make sure 'MAYA_PLUG_IN_PATH' set correctly."); }

// Scene file existence check
string \$scene = `file -q -sn`;
if ( size( \$scene ) == 0 )
    { error ("'"+\$scene+"': scene file not found or not specified\\n\\n"); }

// Initialize Mayatomr globals
miCreateDefaultNodes();

// Init mray globals
//
// Maya 6.x and older                                  // Maya 7.x and up
// ------------------                                  // ---------------
setAttr "mentalrayGlobals.startFrame"     $opt{sfrm};  setAttr "defaultRenderGlobals.startFrame"     $opt{sfrm};
setAttr "mentalrayGlobals.endFrame"       $opt{efrm};  setAttr "defaultRenderGlobals.endFrame"       $opt{efrm};
setAttr "mentalrayGlobals.byFrame"        1;           setAttr "defaultRenderGlobals.byFrame"        1;
setAttr "mentalrayGlobals.startExtension" $opt{sfrm};  setAttr "defaultRenderGlobals.startExtension" $opt{sfrm};
setAttr "mentalrayGlobals.byExtension"    1;           setAttr "defaultRenderGlobals.byExtension"    1;

//// NONE OF THESE WORK. A/W: How do you override the "output" command in the mi file?
////                     Tried mentalrayGlobals.(outp,outputPath,imageFilePath).
////                     Tried Mayatomr (-project,-dir).
////
//// // Image dir specified?
//// if ( "$opt{ImageDirectory}" != "-" )
//// {
////     // Force Mayatomr to use our ImageDirectory. -erco 02/09/04
////     setAttr "mentalrayGlobals.outp" -type "string" "$opt{ImageDirectory}";
////     setAttr "mentalrayGlobals.outputPath" -type "string" "$opt{ImageDirectory}";
////     setAttr "mentalrayGlobals.imageFilePath" -type "string" "$opt{ImageDirectory}";
//// }

// Generate .mi file. 
if ( "$opt{Debug}" == "off" )
{
    // Use -binary if not debugging.
    Mayatomr -mi -binary -padframe 4 -perframe 2
             -verbosity 5 -file "$opt{mipath}" $opt{MayatomrFlags};
}
else 
{
    // No -binary if debugging.
    Mayatomr -mi -padframe 4 -perframe 2 
             -verbosity 5 -file "$opt{mipath}" $opt{MayatomrFlags};
}

//////////////////////////////////////////////////////// MRAY STANDALONE MEL SCRIPT: END

EOF
    close(MEL);
    return(0);
}

# RENDER USING MENTAL RAY IN STANDALONE MODE
#     This generates an .mi file, then renders it with the
#     external version of mental ray, avoiding the need for a GUI license.
#     Returns exit codes: 0 if ok, 1 on error, 2 to requeue
#     (May exit on its own with 2 to force requeue.)
#
sub RenderMrayStandalone(%)
{
    my (%opt) = @_;
    my $melscript = "$ENV{RUSH_TMPDIR}/foo.mel";	# path to tmp mel script
    $opt{mipath} = "$ENV{RUSH_TMPDIR}/foo.mi";

    # PRINT WHAT WE'RE RENDERING
    print STDERR "\n*** MAYA MENTAL RAY STANDALONE ***\n".
		 "    RENDERER: $opt{Renderer}\n".
		 "   MAYAFLAGS: $opt{MayaFlags}\n".
                 "   SCENEPATH: $opt{ScenePath}\n".
		 "     PROJECT: $opt{Project}\n".
		 "    IMAGEDIR: $opt{ImageDirectory}\n".
		 " BATCHFRAMES: $opt{BatchFrames} ($opt{sfrm}-$opt{efrm})\n".
		 "   MRAYFLAGS: $opt{MrayFlags}\n".
		 " MRAYVERBOSE: $opt{MrayVerbose}\n".
		 "   SAVEMIDIR: $opt{SaveMIDirectory}\n".
		 "     RETRIES: $opt{Retries} ".
			 "($opt{RetryBehavior} after $opt{Retries} retries)\n".
		 "  MAXLOGSIZE: $opt{MaxLogSize}\n".
		 "        PATH: $ENV{PATH}\n";

    # Create a temporary mel script used to generate the mi files
    CreateMrayStandaloneMel($melscript, %opt);

    # DEBUGGING: PRINT OUT THE MEL SCRIPT
    print CatFile($melscript);

    # INVOKE MAYA MI GENERATION PASS
    print STDERR "*** MI GENERATION PASS ***\n";

    # MAYA RENDER COMMAND
    my $command = (($G::iswindows) ? "mayabatch " : "maya -batch ") .
		  "-proj $opt{Project} ".
		  "-script $melscript ".
		  "$opt{MayaFlags}".
		  "-file $opt{ScenePath}";

    # HANDLE TRIMMING LOG SIZE
    if ( $opt{MaxLogSize} > 0 || $G::logtrim_always )
	{ $command = "logtrim -s $opt{MaxLogSize} -c $command"; }

    unless ( chdir($opt{Project}) )
        { print "chdir($opt{Project}): $!\n"; return(1); }

    my $err = RunMaya($command, "MAYA");
    if ( $err != 0 ) { return($err); }

    # DEBUGGING: SHOW MI FILE IN LOG
    if ( $opt{Debug} eq "on" )
    {
        my $mifile = sprintf("$ENV{RUSH_TMPDIR}/foo.%04d.mi", $opt{sfrm});
	print "------------------------------- MI FILE: START ($mifile)\n".
	      CatFile($mifile)."\n".
	      "------------------------------- MI FILE: END\n\n";
    }

    ### XXX: REPLACE THIS IF RAY'S -file_name/-file_type "rgb|rla|etc" WORKS
    ### XXX: Better to leave cwd set to the temp dir so junk is cleaned up.
    if ( $opt{ImageDirectory} ne "-" )
    {
	unless ( chdir($opt{ImageDirectory}) )
        {
	    print "FAIL: chdir($opt{ImageDirectory}): $!\n"; 
	    exit(1); 
	}
    }

    # Render all frames in the batch
    print STDERR "\n*** MENTAL RAY RENDER PASS ***\n";
    my $frm;
    for ( $frm = $opt{sfrm}; $frm <= $opt{efrm}; $frm++ )
    {
        my $mifile = sprintf("$ENV{RUSH_TMPDIR}/foo.%04d.mi", $frm);

	# SAVE A COPY OF MI FILE?
	if ( $opt{SaveMIDirectory} ne "-" )
	{
	    print STDERR "--- Copying $mifile to $opt{SaveMIDirectory}: ";
	    use File::Copy;
	    unless ( copy($mifile, $opt{SaveMIDirectory}) )
	        { print STDERR "FAILED\nWARNING: copy failed: $!\n"; }
	    else
	        { print STDERR "OK\n"; }
	}

	# RENDER COMMAND
	$command = "ray $opt{MrayFlags} ". 
	           "-verbose $opt{MrayVerbose} ".
	           "$mifile";

###	# XXX: THIS WOULD BE BETTER THAN chdir() IF DIDN'T HAVE TO SPECIFY EXTENSION
###	#      According to MRmanual docs, -file_name needs an extension.
###	#
###	if ( $opt{ImageDirectory} ne "-" )
###        {
###	    unless ( chdir($opt{Project}) )
###		{ print "chdir($opt{Project}): $!\n"; return(1); }
###	
###	    my $sceneprefix = $opt{ScenePath};
###	    $sceneprefix =~ s%.*[/\\]%%g;	# "/somepath/foo.ma" -> "foo.ma"
###	    $sceneprefix =~ s%\..*%%g;		# "foo.ma" -> "foo"
###	    $command .= sprintf(" -file_name $opt{ImageDirectory}/$sceneprefix.%04d",
###				$frm);
###	}

	if ( $opt{MaxLogSize} > 0 || $G::logtrim_always )
	    { $command = "logtrim -s $opt{MaxLogSize} -c $command"; }

	my $err = RunMaya($command, "MRAY-STANDALONE");
	if ( $err != 0 ) { return($err); }
    }
    return(0);					# all done OK
}

# RENDER WITH MAYA
#     Returns exit codes: 0 if ok, 1 on error, 2 to requeue
#     (May exit on its own with 2 to force requeue.)
# 
sub RenderMaya(%)
{
    my (%opt) = @_;
    # PRINT WHAT WE'RE RENDERING
    print STDERR "\n*** MAYA RENDER ***\n".
		 "   RENDERER: $opt{Renderer}\n".
		 "  MAYAFLAGS: $opt{MayaFlags}\n".
                 "  SCENEPATH: $opt{ScenePath}\n".
		 "    PROJECT: $opt{Project}\n".
		 "   IMAGEDIR: $opt{ImageDirectory}\n".
		 "BATCHFRAMES: $opt{BatchFrames} ($opt{sfrm}-$opt{efrm})\n".
		 "    RETRIES: $opt{Retries} ".
			 "($opt{RetryBehavior} after $opt{Retries} retries)\n".
		 " MAXLOGSIZE: $opt{MaxLogSize}\n".
		 "       PATH: $ENV{PATH}\n";

    # MAYA RENDER COMMAND
    my $command = (($G::iswindows) ? "mayabatch " : "maya -batch ") .
		    "-render " .
		    (($opt{MayaVerbose} eq "on") ? "-verbose 1 " : "") .
		    "-proj $opt{Project} " .
		    "-s $opt{sfrm} ".
		    "-e $opt{efrm} ".
		    "-b 1 " .
		    "$opt{MayaFlags}".
		    "$opt{ImageDirectoryFlags}".
		    "$opt{ScenePath}";

    # HANDLE TRIMMING LOG SIZE
    if ( $opt{MaxLogSize} > 0 || $G::logtrim_always )
	{ $command = "logtrim -s $opt{MaxLogSize} -c $command"; }

    my $err = RunMaya($command, "MAYA");
    if ( $err != 0 ) { return($err); }

    return(0);
}

# RESET THE CURRENT FORM TO THE DEFAULTS -- HANDLES "-defaults"
#    Rewrite defaults, reload form.
#
sub MAIN_Defaults()
{
    my $errmsg = WriteDefaults($ENV{INPUT_DBASE});
    if ( $errmsg ne "" ) { ErrorWindow($errmsg); exit(1); }
    exit(0);
}

# DISPLAY RENDERED IMAGE -- HANDLES "-imgcommand"
#      TO BE DONE: 
#      Need someone to send sample shake output in order
#      to determine how to determine the rendered image's pathname.
#
sub MAIN_ImageCommand()
{
    my $frame   = $ARGV[1];		# frame number user clicked on
    my $logfile = GetLogFile($frame);	# logfile for that frame

    # DOES LOG FILE EXIST?
    if ( ! -e $logfile ) 
    {
	ErrorWindow("**** No logfile yet ****");
	exit(1);
    }

    # PARSE RENDERED IMAGE PATHNAME FROM LOGFILE
    my $image = GetRenderedImage($logfile);

    # LOCAL OS WINDOWS, PATHNAME UNIX STYLE? FORCE UNC (XRAY/A52 07/22/05)
    #    eg. /server/foo -> //server/foo
    #
    if ( $G::iswindows && ( $image !~ m%^//% && $image !~ m%^\\\\% && $image !~ m%^[A-z]:% ) )
	{ $image = "/$image"; }

    if ( $image eq "-" )
    {
	ErrorWindow("**** No finished render ****\n".
		    "\n".
		    "The renderer has not printed a line in the log file\n".
		    "indicating where the image was written.\n".
		    "The render might still be in progress\nor it failed.\n");
	exit(1);
    }

    DisplayImage($image, "fcheck");
    exit(0);
}

# PRESENT USER WITH INPUT FORM
#    Prompts the user for data. 
#
sub MAIN_Input()
{
    # CREATE DEFAULTS IF NONE
    if ( ! -e $G::lastsubmit )
    {
        if ( CreateDotRushDir() < 0 ) { exit(1); }
        my $errmsg = WriteDefaults($G::lastsubmit);
	if ( $errmsg ne "" ) { ErrorWindow($errmsg); exit(1); }
    }

    # CREATE HISTORY FILE IF NONE
    open(FD, ">>${G::histfile}"); close(FD);

    # REPLACE LEGACY KEYWORDS WITH NEW IN HISTORY FILES.
    #    Lets new version parse old history files.
    #
    {
        # Convert keynames: 102.41 -> 102.42 
	my $errmsg;
	my $regex = 's/^[\s]*ImgCommand:/ImageCommand:/;
		     s/^[\s]*SubmitHost:/SubmitHosts:/;
		     s/^[\s]*ImgDirectory:/ImageDirectory:/;
		     s/^[\s]*ImageDir:/ImageDirectory:/;
		     s/^[\s]*LogDir:/LogDirectory:/;
		     s/^[\s]*NeverCpus:/NeverUseCpus:/;';

	if ( ApplyRegexToFile($G::histfile, $regex, \$errmsg) < 0 )
	    { print "WARNING: Couldn't repair histfile: $errmsg\n"; }

	if ( ApplyRegexToFile($G::lastsubmit, $regex, \$errmsg) < 0 )
	    { print "WARNING: Couldn't repair lastsubmit: $errmsg\n"; }
    }

    # PRESENT INPUT FORM TO USER
    #    Database names for each field are derived by removing 
    #    all spaces from the prompt name. Docs for each field should
    #    be placed in examples/help.
    #
    exit(PresentInputForm(<<"EOF"));

<<TITLE>> "Rush Maya Submit"
<<HEADING>> "Rush Maya Submit"
<<SAVE-ID>> "Maya-Submit"

<<TAB>> "Main"



          Job Title: ______________ ?
         Scene Path: ____________________________________________ ? Browse"*.{ma,mb}"
           Renderer: "maya,mentalray,mentalray-standalone,vector" ?



=            Frames: ______________ ?
=      Batch Frames: ______________ ?    |=         Batch Clip: "yes,no" ?



           Max Time: "00:00:00,00:05:00,00:10:00,00:30:00,01:00:00,01:30:00,02:00:00,02:30:00,03:00:00,_" ?
       MaxTimeState: "Que,Fail,Done,Hold" ?


               Cpus: ___________________________________________________ ?
                     ___________________________________________________
                     ___________________________________________________


<<SUBMITDEFAULTS>>

<<ENDPAGE>>
<<TAB>> "Maya"
<<SUBHEADING-LEFT>> "Maya"
       Maya Verbose: "on,off" ?
         Maya Flags: __________________________________________________ ?

<<SUBHEADING-LEFT>> "Mayatomr"
   Mayatomr Verbose: "0,1,2,3,4,5" ?
     Mayatomr Flags: __________________________________________________ ?

<<SUBHEADING-LEFT>> "Mental Ray Standalone"
       Mray Verbose: "0,1,2,3,4,5" ?
         Mray Flags: __________________________________________________ ?
  Save MI Directory: __________________________________________________ ?

<<SUBHEADING-LEFT>> "Vector"
    Vectorize Flags: __________________________________________________ ?

<<TAB>> "Advanced"

    Image Directory: ___________________________________________ ? Browse
      Image Command: ___________________________________________ ? Browse


=     Log Directory: ___________________________________________ ? Browse
=         Log Flags: "Keep Last,Keep All,Overwrite" ?
=      Max Log Size: ______________ ?


           AutoDump: "off,fail,done,donefail" ?
          Done Mail: ______________ ?
          Dump Mail: ______________ ?


=           WaitFor: ______________ ?
=    Wait For State: "Dump,Done,DoneFail,Fail" ?


                Ram: ______________ ?
       Submit Hosts: __________________________________________________ ?


  Job Start Command: ___________________________________________ ? Browse
   Job Done Command: ___________________________________________ ? Browse
   Job Dump Command: ___________________________________________ ? Browse


     Never Use Cpus: __________________________________________________ ?
                     __________________________________________________
                     __________________________________________________

     Submit Options: __________________________________________________ ?
                     __________________________________________________
                     __________________________________________________


=       Start Irush: "yes,no,ask" ?  |= Print Environment?: "off,on" ?
=           Retries: "1,2,3,4,5,_" ? |=     Retry Behavior: "Fail,AddNever+Requeue" ?
=             Debug: "off,on" ?


EOF
}

# SUBMIT THE JOB -- HANDLES "-submit"
#     Parse the user's input, and submit the job to rush.
#     $ARGV[1]: input filename containing key/value pairs of user's input
#
sub MAIN_Submit()
{
    # LOAD INPUT FORM DATA INTO %in, SAVE COPY TO HISTORY FILE
    my %in;
    my $errmsg;
    if ( LoadInput(\%in, $ARGV[1], \$errmsg) < 0 )
	{ print "$errmsg\n"; exit(1); }
    if ( SaveInput(\%in, ${G::lastsubmit}, \$errmsg) < 0 )
	{ print "$errmsg\n"; exit(1); }

    # PARSE STANDARD SUBMIT OPTIONS
    my $submitoptions = ParseStandardSubmit(\%in);
    my $submithost = PickOneSubmitHost($in{SubmitHosts});

    # BATCH FRAMES
    my ($batchframes, $batchend, $newframespec) = BatchFramesSubmit(\%in);

    # SANITY
    $in{ImageDirectory} ||= "-";

    # PATH FIXUPS
    $in{ScenePath} = FixPath($in{ScenePath});
    $in{ImageDirectory} = ( $in{ImageDirectory} ne "-" ) 
				? FixPath($in{ImageDirectory}) : "-";

    # PARSE PROJECT
    #     Everything to the left of 'renderscenes' (or whatever
    #     'project separators' were specified) is used as the 
    #     project path. Case is ignored.
    #
    my $project = "-";
    foreach ( @G::projsep )
    {
	if ( $in{ScenePath} =~ m%(^.*)$_%i )
	    { $project = $1; last; }
    }
    if ( $project eq "" || $project eq "-" )
    {
	print STDERR "Could not parse project from path:\n" .
		     "$in{ScenePath}\n" .
		     "See the '?' help docs for the 'Scene File' prompt.\n";
	exit(1);
    }

    # JOB TITLE
    $in{JobTitle} =~ s/[ \t][ \t]*/_/g;	# convert embedded white to underbars
    if ( $in{JobTitle} eq "-" || $in{JobTitle} eq "" )
    {
	# PARSE TITLE FROM FIRST 25 CHARACTERS OF SCENE FILENAME
	if ( $in{ScenePath} =~ m%/([^./]*)\.m[ab]%i )
	{
	    $in{JobTitle} = substr($1, 0, 25);		# first 25 chars
	    $in{JobTitle} =~ tr/[a-z]/[A-Z]/;		# uppercase
	    $in{JobTitle} =~ s/\..*//;			# "foo.%04d.mi" -> "foo"
	}
    }
    $in{JobTitle}        ||= "SUBMIT-MAYA";

    $in{MayaFlags}       ||= "-";
    $in{MayatomrFlags}   ||= "-";
    $in{VectorizeFlags}  ||= "-";
    $in{MrayFlags}       ||= "-";
    $in{SaveMIDirectory} ||= "-";
    $in{Debug}           ||= "off";

    # LOGDIR BASED ON SCENE PATH
    if ( $in{LogDirectory} eq "" )
    {
	$in{LogDirectory} = "$in{ScenePath}.log";

	# CREATE LOGDIR ONLY IF AUTOMATICALLY GENERATED
	if ( ! -d $in{LogDirectory} )
	{
	    unless ( mkdir($in{LogDirectory}, 0777) )
	    {
		print STDERR "mkdir($in{LogDirectory}): $!\n" .
			    "Can't make log directory.\n";
		exit(1);
	    }
	    # WINDOWS: OPEN ACLS FOR THE LOG DIRECTORY
	    if ( $G::iswindows )
	    {
		my $dirname = $in{LogDirectory}; $dirname =~ s%/%\\%g;
		system("cacls $dirname /e /c /g everyone:f");
	    }
	}
    }
    if ( $in{LogDirectory} ne "-" )
    {
	$in{LogDirectory} = FixPath($in{LogDirectory});
	$submitoptions .= "logdir          $in{LogDirectory}\n";
    }

    # MAX LOG SIZE SANITY
    if ( $in{MaxLogSize} <= 0 )
	{ $in{MaxLogSize} = 0; }

    # CONVERT SPACES TO DEL
    #    Do this so spaces pass across command lines correctly.
    #
    $in{MayaFlags}       =~ s/ /\x7f/g;
    $in{MayatomrFlags}   =~ s/ /\x7f/g;
    $in{VectorizeFlags}  =~ s/ /\x7f/g;
    $in{MrayFlags}       =~ s/ /\x7f/g;
    $in{SaveMIDirectory} =~ s/ /\x7f/g;

    # RETRIES
    if ( $in{Retries} eq "" || $in{Retries} eq "-" || $in{Retries} == 0 )
	{ $in{Retries} = 1; }

    if ( $in{RetryBehavior} eq "" || $in{RetryBehavior} eq "-" )
	{ $in{RetryBehavior} = "Fail"; }

    my $submit = <<"EOF";
title	        $in{JobTitle}
frames          $newframespec
command         perl $G::self -render $batchframes $batchend $in{Retries} $in{RetryBehavior} $in{ScenePath} $project $in{ImageDirectory} $in{Renderer} $in{MayaVerbose} $in{MayatomrVerbose} $in{MaxLogSize} $in{PrintEnvironment} $in{MayaFlags} $in{MayatomrFlags} $in{VectorizeFlags} $in{MrayVerbose} $in{MrayFlags} $in{SaveMIDirectory} $in{Debug}
$submitoptions
EOF
    # ErrorWindow("ERCO DEBUGGING: SUBMIT OPTIONS:\n---\n$submitoptions\n--- ImageCommand: ".(defined($in{ImageCommand})?$in{ImageCommand}:"(UNDEF)"));
    exit(RushSubmit($submithost, $submit, $in{JobTitle}, $in{StartIrush}));
}

# RENDER THE FRAME -- HANDLES "-render"
#   Invoked on each machine to render the frame.
#   Arguments passed via rush from the MAIN_Submit() section.
#
sub MAIN_Render()
{
    my %opt;
    shift(@ARGV);
    $opt{BatchFrames}      = $ARGV[0]; shift(@ARGV);
    $opt{BatchEnd}         = $ARGV[0]; shift(@ARGV);
    $opt{Retries}          = $ARGV[0]; shift(@ARGV);
    $opt{RetryBehavior}    = $ARGV[0]; shift(@ARGV);
    $opt{ScenePath}        = $ARGV[0]; shift(@ARGV);
    $opt{Project}          = $ARGV[0]; shift(@ARGV);
    $opt{ImageDirectory}   = $ARGV[0]; shift(@ARGV);
    $opt{Renderer}         = $ARGV[0]; shift(@ARGV);
    $opt{MayaVerbose}      = $ARGV[0]; shift(@ARGV);
    $opt{MayatomrVerbose}  = $ARGV[0]; shift(@ARGV);
    $opt{MaxLogSize}       = $ARGV[0]; shift(@ARGV);
    $opt{PrintEnvironment} = $ARGV[0]; shift(@ARGV);
    $opt{MayaFlags}        = $ARGV[0]; shift(@ARGV);
    $opt{MayatomrFlags}    = $ARGV[0]; shift(@ARGV);
    $opt{VectorizeFlags}   = $ARGV[0]; shift(@ARGV);
    $opt{MrayVerbose}      = $ARGV[0]; shift(@ARGV);
    $opt{MrayFlags}        = $ARGV[0]; shift(@ARGV);
    $opt{SaveMIDirectory}  = $ARGV[0]; shift(@ARGV);
    $opt{Debug}            = $ARGV[0]; shift(@ARGV);

    # REMOVE DOUBLE SLASHES FROM LINUX
    #    Customers indicated UNCs (//) in scene or project path
    #    cause weird hangs under linux w/Maya 5.0 eg. if texure
    #    maps are not found (maya hangs using 99% of the cpu)
    #
    if ( ! $G::iswindows )
    {
	$opt{ScenePath}      =~ s%//%/%g;
	$opt{Project}        =~ s%//%/%g;
	$opt{ImageDirectory} =~ s%//%/%g;
    }

    # Massage opt{} values
    $opt{MayaFlags}      =~ s/\x7f/ /g; 			# DEL->SPACE
    $opt{MayaFlags}      =~ s/\$ENV{(\w+)}/$ENV{$1}/g;	# expand embedded perl var
    $opt{MayaFlags}      = ( $opt{MayaFlags} eq "-" ) ? "" : "$opt{MayaFlags} ";
    $opt{MayatomrFlags}  =~ s/\x7f/ /g;			# DEL->SPACE
    $opt{MayatomrFlags}  = ( $opt{MayatomrFlags} eq "-" ) 
				    ? "" : "$opt{MayatomrFlags} ";
    $opt{VectorizeFlags} =~ s/\x7f/ /g;			# DEL->SPACE
    $opt{VectorizeFlags} = ( $opt{VectorizeFlags} eq "-" ) 
				    ? "" : "$opt{VectorizeFlags} ";
    $opt{MrayFlags}      =~ s/\x7f/ /g;			# DEL->SPACE
    $opt{MrayFlags}      = ( $opt{MrayFlags} eq "-" ) ? "" : "$opt{MrayFlags} ";

    $opt{SaveMIDirectory} =~ s/\x7f/ /g;		# DEL->SPACE

    $opt{ImageDirectoryFlags} = ( $opt{ImageDirectory} eq "-" ) 
				    ? "" : "-rd $opt{ImageDirectory} ";

    # BATCH FRAMES
    ($opt{sfrm}, $opt{efrm}) = BatchFramesRender($ENV{RUSH_FRAME},
					  $opt{BatchFrames}, $opt{BatchEnd});

    # MAYA WILL USE RUSH_TMPDIR FOR TEMP FILES
    #    For more info, see rush.general newsgroup posting 11/09/2004, 03:05pm
    #
    $ENV{TEMP} = $ENV{TEMPDIR} = $ENV{RUSH_TMPDIR};		# maya uses TEMP and TEMPDIR
    $ENV{TMPDIR} = $ENV{RUSH_TMPDIR};				# and this just in case

    # PRINT ENVIRONMENT?
    if ( $opt{PrintEnvironment} eq "on" )
    {
	print "\n****** ENVIRONMENT: START\n";
	system( ($G::iswindows) ? "set" : "printenv|sort" );
	print "****** ENVIRONMENT: END\n";
    }

    # MAYA RENDER RETRY LOOP
    my $try;
    for ( $try=1; $try<=$opt{Retries}; $try++ )
    {
	if ( $try > 1 )
	{
	    print "\n--- RETRY #$try of $opt{Retries}\n";
	    system("rush -fu -notes $ENV{RUSH_FRAME}:".
		   "\"retry#$try of $opt{Retries}\"");
	    sleep(10);
	}

	# USE APPROPRIATE RENDERER
	my $err = 0;
	if ( $opt{Renderer} eq "mentalray" )
	    { $err = RenderMray(%opt); }
	elsif ( $opt{Renderer} eq "mentalray-standalone" )
	    { $err = RenderMrayStandalone(%opt); }
	elsif ( $opt{Renderer} eq "vector" )
	    { $err = RenderVector(%opt); }
	elsif ( $opt{Renderer} eq "maya" )
	     { $err = RenderMaya(%opt); }
	else
	    { print STDERR "UNKNOWN RENDERER: '$opt{Renderer}'\n"; exit(1); }

	# RENDERED OK? DONE
	if ( $err == 0 )
	    { exit(0); }
    }

    if ( $opt{RetryBehavior} eq "AddNever+Requeue" )  	# REQUEUE/FAIL
    { 
	print "--- FAILED: [$ENV{RUSH_HOSTNAME}] AddNever+Requeue\n";
	system("rush -fu -an $ENV{RUSH_HOSTNAME}");	# NEVERHOST
	exit(2);					# REQUEUE
    }

    print "--- FAILED\n";
    exit(1);
}

###
### MAIN
###
{
    # Silence 'variable used once' warnings
    $G::title    .= "";
    $G::progname .= "";
    $G::iswindows = $G::iswindows;
    $G::ismac     = $G::ismac;
    $G::islinux   = $G::islinux;
    $G::isirix    = $G::isirix;
    $G::self      = $G::self;

    # PARSE COMMAND LINE
    if ( ! defined ( $ARGV[0] ) )
        { MAIN_Input(); }
    elsif ( $ARGV[0] eq "-submit" )
        { MAIN_Submit(); }
    elsif ( $ARGV[0] eq "-render" )
        { MAIN_Render(); }
    elsif ( $ARGV[0] eq "-imgcommand" )
        { MAIN_ImageCommand(); }
    elsif ( $ARGV[0] eq "-defaults" )
        { MAIN_Defaults(); }
    else
        { ErrorWindow("'$ARGV[0]': unknown argument"); exit(1); }
    exit(0);		# just in case
}

